import React, { useState } from 'react';
import { X, Plus, Trash } from 'lucide-react';

export function SettingsModal({ onClose, settings, onSave }) {
    const [keys, setKeys] = useState(settings.apiKeys || []);
    const [model, setModel] = useState(settings.modelConfig?.model || 'gemini-1.5-flash');

    const addKey = () => setKeys([...keys, '']);
    const removeKey = (idx) => setKeys(keys.filter((_, i) => i !== idx));
    const updateKey = (idx, val) => {
        const newKeys = [...keys];
        newKeys[idx] = val;
        setKeys(newKeys);
    };

    const handleSave = () => {
        const validKeys = keys.filter(k => k.trim() !== '');
        onSave({
            apiKeys: validKeys,
            modelConfig: { ...settings.modelConfig, model }
        });
        onClose();
    };

    return (
        <div className="fixed inset-0 bg-black/50 flex items-center justify-center z-50">
            <div className="bg-[var(--bg-secondary)] w-[500px] rounded-lg shadow-xl border border-[var(--border-color)]">
                <div className="p-4 border-b border-[var(--border-color)] flex justify-between items-center">
                    <h2 className="text-lg font-bold">Settings</h2>
                    <button onClick={onClose}><X size={20} /></button>
                </div>

                <div className="p-4 overflow-y-auto max-h-[70vh]">
                    <div className="mb-6">
                        <label className="block text-sm font-medium mb-2">Gemini Model</label>
                        <select
                            value={model}
                            onChange={(e) => setModel(e.target.value)}
                            className="w-full bg-[var(--bg-tertiary)] border border-[var(--border-color)] rounded p-2 text-white outline-none focus:border-[var(--accent-primary)]"
                        >
                            <option value="gemini-flash-latest">Gemini Flash latest(Fast)</option>
                            <option value="gemini-2.5-pro">Gemini 2.5 Pro(Fast)</option>
                            <option value="gemini-1.5-pro">Gemini 1.5 Pro (Better Quality)</option>
                            <option value="gemini-1.0-pro">Gemini 1.0 Pro</option>
                        </select>
                    </div>

                    <div className="mb-4">
                        <label className="block text-sm font-medium mb-2">API Keys (Round Robin)</label>
                        <div className="space-y-2">
                            {keys.map((k, i) => (
                                <div key={i} className="flex gap-2">
                                    <input
                                        type="text"
                                        value={k}
                                        onChange={(e) => updateKey(i, e.target.value)}
                                        className="flex-1 bg-[var(--bg-tertiary)] border border-[var(--border-color)] rounded p-2 text-white text-sm"
                                        placeholder="AIzaSy..."
                                    />
                                    <button onClick={() => removeKey(i)} className="text-[var(--accent-error)] hover:bg-[var(--bg-primary)] p-2 rounded"><Trash size={16} /></button>
                                </div>
                            ))}
                        </div>
                        <button onClick={addKey} className="mt-2 text-sm text-[var(--accent-primary)] flex items-center gap-1 hover:underline">
                            <Plus size={14} /> Add API Key
                        </button>
                    </div>
                </div>

                <div className="p-4 border-t border-[var(--border-color)] flex justify-end gap-2">
                    <button onClick={onClose} className="btn">Cancel</button>
                    <button onClick={handleSave} className="btn btn-primary">Save Changes</button>
                </div>
            </div>
        </div>
    );
}
